<?php
/*
Plugin Name: SearchLink
Version: 0.5
Plugin URI: http://vince.tikasz.hu/searchlink/
Description: Replace google://search+me strings to links to search result for 'search me' on google. Enable search engines on <a href="options-general.php?page=searchlink.php">Searchlink option</a> page.
Author: Tikász Vince
Author URI: http://vince.tikasz.hu
*/

define( 'SEARCHLINK_DEBUG', false );

/**
 * Searclink Class
 *
 * @since 0.3
 */
class Searchlink
{
	/**
	 * @since 0.4.5
	 * @var String
	 */
	var $_version = '0.5';

	/**
	 * @since 0.3
	 * @var Array $options
	 */
	var $options;

	/**
	 * Display debug messages?
	 * @since 0.3
	 * @var Bool $debug
	 */
	var $debug;

	/**
	 * Debug messages
	 * @since 0.3
	 * @var String $debug_msg
	 */
	var $debug_msg;

	/**
	 * Engines
	 * @since 0.3
	 * @var Array
	 * @private
	 */
	var $engines;

	/**
	 * Protocolls
	 * @since 0.3
	 * @var Array
	 * @private
	 */
	var $protocolls;

	/**
	 * Alliases
	 * @since 0.3
	 * @var Array
	 * @private
	 */
	var $aliases;
	
	/**
	 * Cunstructor
	 * @since 0.3
	 * @return void
	 */
	function Searchlink()
	{
		$this->debug = SEARCHLINK_DEBUG;
		$this->options = get_option('searchlink_options');
		$version = get_option( 'searchlink_version' );
		if ( $version != $this->_version )
		{
			$this->_doUpdate( $version );
		}

		$storedEngines = $this->_getEngines();

		foreach ( $storedEngines as $name => $engine )
		{
			if (
				$this->is_engine( $engine )
				&& isset( $engine['enabled'] )
				&& true == $engine['enabled']
			)
			{
				$this->addEngine( $engine );
			}
		}
	}
	
	/**
	 * Default Options for reset optoions and for reset
	 * @since 0.3.5
	 * @return Array
	 */
	function getDefaultOptions()
	{
		return array(
			'engine' => array(
				'sitesearch' => true, 
				'google' => true, 
				'google-image' => true, 
				'yahoo' => true, 
				'yahoo-image' => true, 
				'msn' => true, 
				'msn-image' => true, 
				'altavista' => true, 
				'altavista-image' => true, 
				'alltheweb' => true, 
				'alltheweb-image' => true, 
				'miner' => true, 
			),
			'engines' => array (
				'sitesearch' => array ( 'name' => 'sitesearch', 'enabled' => true, 'engine_url' => '/?s=','label' => 'Site','aliases' => array ( 'itt', 'here' ),'image' => false,'extra-class' => ''),
				'google' => array ( 'name' => 'google', 'enabled' => true,'engine_url' => 'http://www.google.com/search?q=','label' => 'Google','aliases' => array(),'image' => false,'extra-class' => ''),
				'google-image' => array ( 'name' => 'google-image', 'enabled' => true,'engine_url' => 'http://images.google.com/images?q=','label' => 'Google Image','aliases' => array ('google-images','google-picture','google-pictures'),'image' => true,'extra-class' => 'google'),
				'yahoo' => array ( 'name' => 'yahoo', 'enabled' => true,'engine_url' => 'http://search.yahoo.com/search?p=','label' => 'Yahoo!','aliases' => array(),'image' => false,'extra-class' => ''),
				'yahoo-image' => array ( 'name' => 'yahoo-image', 'enabled' => true,'engine_url' => 'http://images.search.yahoo.com/search/images?p=','label' => 'Yahoo! Image','aliases' => array('yahoo-images','yahoo-picture','yahoo-pictures'),'image' => true,'extra-class' => 'yahoo'),
				'msn' => array( 'name' => 'msn', 'enabled' => true,'engine_url' => 'http://search.live.com/results.aspx?q=','label' => 'MSN','aliases' => 'live','image' => false,'extra-class' => ''),
				'msn-image' => array ( 'name' => 'msn-image', 'enabled' => true,'engine_url' => 'http://search.live.com/images/results.aspx?q=','label' => 'MSN Image','aliases' => array('live-image','msn-image','msn-images','live-picture','msn-picture','live-pictures','msn-pictures'),'image' => true,'extra-class' => 'msn'),
				'altavista' => array( 'name' => 'altavista', 'enabled' => true,'engine_url' => 'http://www.altavista.com/web/results?q=','label' => 'Altavista','aliases' => array(),'image' => false,'extra-class' => ''),
				'altavista-image' => array( 'name' => 'altavista-image', 'enabled' => true,'engine_url' => 'http://www.altavista.com/image/results?q=','label' => 'Altavista Image','aliases' => array('altavista-images','altavista-picture','altavista-pictures'),'image' => true,'extra-class' => 'altavista'),
				'alltheweb' => array( 'name' => 'alltheweb', 'enabled' => true,'engine_url' => 'http://www.alltheweb.com/search?q=','label' => 'AllTheWeb','aliases' => array(),'image' => false,'extra-class' => ''),
				'alltheweb-image' => array( 'name' => 'alltheweb-image', 'enabled' => true,'engine_url' => 'http://www.alltheweb.com/search?cat=img&amp;q=','label' => 'AllTheWeb Image','aliases' => array('alltheweb-images','alltheweb-picture','alltheweb-pictures'),'image' => true,'extra-class' => 'alltheweb'),
				'miner' => array( 'name' => 'miner', 'enabled' => true,'engine_url' => 'http://miner.hu/?q=','label' => 'Miner','aliases' => array(),'image' => false,'extra-class' => 'miner')
			)
		);
	}


	/**
	 * Register Activation
	 * @since 0.3.3
	 * @return void
	 */
	function register_activation()
	{
		$options = $this->getDefaultOptions();
		$engines = $options['engines'];
		unset( $options['engines'] );
		add_option('searchlink_options', $options);
		add_option('searchlink_engines', $engines);
		$searchlink_options = get_option('searchlink_options');
		foreach ( $options as $key => $val )
		{
			if (!isset($searchlink_options[$key]))
			{
				$searchlink_options[$key] = $options[$key];
			}
		}
		update_option('searchlink_options', $searchlink_options);
	}

	/**
	 * Get engine datas
	 * @since 0.5
	 * @return Array
	 */
	function _getEngines()
	{
		$engines = get_option('searchlink_engines');
		foreach( $engines as $name => $data )
		{
			//$engines[ $name ];
		}

		return $engines;
	}

	/**
	 * Save engine list
	 * @since 0.5
	 * @return mixed
	 */
	function _setEngines( $engines )
	{
		$_E = array();
		foreach( $engines as $name => $data )
		{
			if (
				!empty( $name )
				&& !empty( $data['name'] )
				&& !empty( $data['label'] )
				&& !empty( $data['engine_url'] )
			)
			{
				$_E[ $name ] = $data;
			}//$engines[ $name ];
		}
		return update_option( 'searchlink_engines', $_E );
	}

	/**
	 * Add an engine
	 * @since 0.3
	 * @param String $name
	 * @param String $engine_url
	 * @param Array $aliases
	 * @param Bool $image
	 * @param String $label @since 0.5
	 * @return void
	 */
	function addEngine( $name, $engine_url = '', $aliases = array(), $image = false, $extra_class = '', $label = null )
	{
		if ( is_array( $name ) )
		{
			$_n = $name;
			if ( ! isset( $_n['name'] ) )
			{
				trigger_error( 'Searclink::addEngine() needs name for new engine', E_USER_ERROR );
			}
			else
			{
				$name = $_n['name'];
			}

			if ( ! isset( $_n['engine_url'] ) )
			{
				trigger_error( 'Searclink::addEngine() needs engine_url for new engine', E_USER_ERROR );
			}
			else
			{
				$engine_url = $_n['engine_url'];
			}
			$aliases = isset( $_n['aliases'] ) ? (array)$_n['aliases'] : array();
			$image = isset( $_n['image'] ) ? $_n['image'] : false;
			$extra_class = isset( $_n['extra_class'] ) ? $_n['extra_class'] : '';
			$label = isset( $_n['label'] ) ? $_n['label'] : null;
		}
		if ( is_null( $label ) )
		{
			$label = $name;
		}
		$this->engines[ $name ] = array(
			'name' => $name,
			'engine_url' => $engine_url,
			'label' => $label,
			'aliases' => $aliases,
			'image' => $image,
			'extra-class' => $extra_class
		);
		$this->protocolls[ $engine_url ] = array( $name );
		$this->aliases[ $name ] = $name;
		foreach( (array)$aliases as $alias )
		{
			if ( !empty( $alias ) )
			{
				$this->protocolls[ $engine_url ][] = $alias;
				$this->aliases[ $alias ] = $name;
			}
		}
	}

	/**
	 * Replace Callback to apply filter on content
	 * @since 0.3
	 * @param Array $matches
	 * @return String
	 */
	function replace_callback( $matches )
	{
		//global $Searchlink;
		$engine_name = $this->aliases[ $matches[2] ];
		$engine = $this->engines[ $engine_name ];

		$class = array( 'search', $engine_name );
		if ( isset( $engine['image'] ) && true == $engine['image'] )
		{
			$class[] = 'search-image';
		}
		if ( !empty(  $engine['extra-class'] ) )
		{
			$class[] = trim( $engine['extra-class'] );
		}
		
		if ( true == $this->debug )
		{
			$this->debug_msg .= "\nmatches : ".print_r( $matches, 1 );
			$this->debug_msg .= "\nengine_name : ".print_r( $engine_name, 1 );
			$this->debug_msg .= "\nengine : ".print_r( $engine, 1 );
			$this->debug_msg .= "\nclasss : " .print_r( $class, 1 );
			$this->debug_msg .= "\n---------\n";
		}
		$content = $matches[1];
		if ( 'label' == $this->options['label_from'] )
		{
			$content = preg_replace( '/^.*:\/\//', $this->engines[ $engine_name ]['label'].'://', $content );
		}
		return '<a rel="search" class="'.join( ' ', $class ).'" href="' .$engine['engine_url'] .$matches[3].'">'.$content.'</a>';
	}
	
	/**
	 * Search Link Parser
	 *
	 * convert 'google://search+me+formated anchors to HTML anchors
	 * @since 0.3
	 * @param String $text
	 * @return String
	 */
	function the_content( $text )
	{
		$ret = $text;

		foreach( (array)$this->protocolls as $engine => $protocolls )
		{
			$regex = "/\b(?<!')((".join('|', $protocolls )."):\/\/([^<'\s]*))\b/i";
			
			if ( preg_match( $regex, $ret ) )
			{
				if ( $this->debug == true )
				{
					$this->debug_msg .= "\n- - -\n{$regex}\n";
				}
				$ret = preg_replace_callback(
					$regex,
					array( $this, 'replace_callback' ),
					$ret
				);
			}
		}
		
		if ( $this->debug == true )
		{
			$ret = '<pre>'. htmlentities( print_r( $this ) ).'</pre>'.$ret;
		}
		
		return $ret;
	}


/**************************************************
 * Administrator FUNCTIONS
 *************************************************/

	/**
	 * Register admin menu item
	 * @since 0.3.7
	 * @return void
	 */
	function admin_menu()
	{
		if (version_compare($wp_version, '2.6', '<')) // Using old WordPress
		{
            load_plugin_textdomain('searchlink', 'wp-content/plugins/searchlink/languages');
		}
        else
        {
            load_plugin_textdomain('searchlink', 'wp-content/plugins/searchlink/languages', 'searchlink/languages');
		}
		add_options_page(
			__('Searchlink Options', 'searchlink'),
			__('Searchlink', 'searchlink'),
			5,
			basename(__FILE__),
			array( $this, 'options_page' )
		);
	}
		
	/**
	 * Check given array is an engine description
	 * @since 0.5
	 * @return Bool
	 */
	function is_engine( $engine )
	{
		if (
			!is_array( $engine )
			|| ( isset( $engine[ 'delete_this' ] ) && 'delete_'.$engine[ 'name' ] == $engine[ 'delete_this' ] )
			|| ( empty( $engine[ 'name' ] ) || empty( $engine['label'] ) || empty( $engine['engine_url'] ) )
		)
		{
			return false;
		}
		return true;
		
	}

	/**
	 * Options page
	 * @since 0.3.7
	 * @return void
	 */
	function options_page()
	{
		$Engines = $this->_getEngines();

		if ( ! $this->options )
		{
			$this->register_activation();
		}
		$update = false;
		if ( isset( $_POST['searchlink_update'] ) && 'update' == $_POST['searchlink_update'] )
		{
			foreach( $_POST['serarchlink']['engine'] as $name => $engine )
			{
				if( $this->is_engine( $engine ) )
				{
					$name = $engine['name'];
					$Engines[ $name ]['name'] = $name;
					$Engines[ $name ]['label'] = $engine['label'];
					$Engines[ $name ]['enabled'] = (bool)$engine['enabled'];
					$Engines[ $name ]['engine_url'] = $engine['engine_url'];
					$Engines[ $name ]['aliases'] = preg_split( "/\s+/", $engine['aliases'] );
					$Engines[ $name ]['image'] = (isset( $engine['image'] ) && '1' == $engine['image']);
					$Engines[ $name ]['extra-class'] = $engine['extra-class'];					
					$this->options['engine'][ $name ] = (bool)$engine['enabled'];
				}
				else
				{
					
					unset( $this->options['engine'][ $name ] );
					unset($Engines[ $name ]);
					continue;
				}

			}
			if ( isset(  $_POST['searchlink_protocol_label']) )
			{
				$this->options['label_from'] = ( 'label' == $_POST['searchlink_protocol_label'] ? 'label' : 'name' );
			}

			$this->_setEngines( $Engines );
			update_option('searchlink_options', $this->options );
			$update = true;
			echo '<div class="updated fade" id="message" style="background-color: rgb(255, 251, 204);"><p><strong>' . __('Options saved.') . '</strong></p></div>';
		}

		if ( true == $this->debug )
		{
			echo '<pre>'. var_export( $this->options, 1 ) . '</pre>';
			echo '<pre>'. var_export( $_POST, 1 ) . '</pre>';

		}
		echo '
<div class="wrap">
	<h2>' . __('Searchlink Options', 'searchlink') . '</h2>
	<form name="form1" method="post" action="' . $_SERVER['REQUEST_URI'] . '">
			<input type="hidden" name="searchlink_update" value="update" />
			<table class="form-table">
				<tr valign="top">
					<th scope="row"><strong>'.__('Protocol from', 'searchlink').'</strong></th>
					<td>
						<label>'.__('Engine name','searchlink').'<input type="radio" '.('name' == $this->options['label_from'] ? 'checked="checked"  ' : '').'name="searchlink_protocol_label" value="name" /></label><br/>
						<label>'.__('Engine label','searchlink').'<input type="radio" '.('label' == $this->options['label_from'] ? 'checked="checked"  ' : '').'name="searchlink_protocol_label" value="label" /></label>
					</td>
				</tr>
			</table>
			<table class="widefat">
				<thead>
					<tr>
						<th class="manage-column column-cb check-column"><span title="'.__('Enabled', 'searchlink').'">'.__('En.', 'searchlink').'</span></th>
						<th class="manage-column column-title">'.__('Name', 'searchlink').'/'.__('Aliases', 'searchlink').'</th>
						<th class="manage-column column-title">'.__('Label', 'searchlink').'/'.__('URL', 'searchlink').'</th>
						<!-- <th class="manage-column column-title">'.__('URL', 'searchlink').'</th> -->
						<th class="manage-column column-cb check-column"><span title="'.__('Image Search', 'searchlink').'">'.__('Img.', 'searchlink').'</span></th>
						<th class="manage-column column-title"><span title="'.__('Extra CSS Class', 'searchlink').'">'.__('Class', 'searchlink').'</span></th>
						<th class="manage-column column-title"><span title="'.__('Delete', 'searchlink').'">'.__('Del.', 'searchlink').'</span></th>
					</tr>
				</thead>
				<tbody class="form-table">
';
		foreach ( $Engines as $name => $data )
		{
			echo '
					<tr>
<td><input type="checkbox" name="serarchlink[engine]['.$name.'][enabled]" id="searchlink_'.$name.'_enabled" value="1" ' . ( true == $data['enabled'] ? ' checked="checked"' : '' ) . '/></td>
<td>
	<input type="text" readonly="readonly" name="serarchlink[engine]['.$name.'][name]" id="searchlink_'.$name.'_name" value="'.$name.'"  size="22"/><br/>
	<textarea cols="20" rows="3" name="serarchlink[engine]['.$name.'][aliases]" id="searchlink_'.$name.'_aliases">'.join("\n",(array)$data['aliases']).'</textarea>
</td>
<td>
	<input type="text" name="serarchlink[engine]['.$name.'][label]" id="searchlink_'.$name.'_label" value="'.$data['label'].'" size="25"/><br/>
	<input type="text" name="serarchlink[engine]['.$name.'][engine_url]" id="searchlink_'.$name.'_engine_url" value="'.$data['engine_url'].'" size="40" />
</td>
<!--
<td><input type="text" name="serarchlink[engine]['.$name.'][engine_url]" id="searchlink_'.$name.'_engine_url" value="'.$data['engine_url'].'" size="20" /></td>
-->
<td><input type="checkbox" name="serarchlink[engine]['.$name.'][image]" id="searchlink_'.$name.'_image" value="1" ' . ( true == $data['image'] ? ' checked="checked"' : '' ) . ' /></td>
<td><input type="text" name="serarchlink[engine]['.$name.'][extra_class]" id="searchlink_'.$name.'_extra_class" value="'.$data['extra_class'].'" size="8"/></td>
<td><input type="checkbox" name="serarchlink[engine]['.$name.'][delete_this]" id="searchlink_'.$name.'_delete" value="delete_'.$name.'" /></td>
					</tr>
			';
		}
			echo '
					<tr>
<td><input type="checkbox" name="serarchlink[engine][0][enabled]" id="searchlink_0_enabled" value="1" checked="checked" /></td>
<td>
	<input type="text" name="serarchlink[engine][0][name]" id="searchlink_0_name" size="22"/><br/>
	<textarea cols="20" rows="3" name="serarchlink[engine][0][aliases]" id="searchlink_0_aliases"></textarea>
</td>
<td>
	<input type="text" name="serarchlink[engine][0][label]" id="searchlink_0_label" size="25"/><br/>
	<input type="text" name="serarchlink[engine][0][engine_url]" id="searchlink_0_engine_url" value="http://" size="40" />
</td>
<!--
<td><input type="text" name="serarchlink[engine][0][engine_url]" id="searchlink_0_engine_url" value="http://" size="20" /></td>
-->
<td><input type="checkbox" name="serarchlink[engine][0][image]" id="searchlink_0_image" value="1" /></td>
<td><input type="text" name="serarchlink[engine][0][extra_class]" id="searchlink_0_extra_class" size="8"/></td>
<td><input type="checkbox" name="serarchlink[engine][0][delete_this]" id="searchlink_0_delete" value="delete_0" disabled="disabled"/></td>
					</tr>
			';

		echo '
				</tbody>
			</table>
		<p class="submit">
		<input class="button-primary" type="submit" value="' . __('Save Changes &raquo;', 'searchlink') . '" name="Submit" >
		</p>
	</form>
</div>
';
	}

/**************************************************
 * Install FUNCTIONS
 *************************************************/

	/**
	 * Update options and stored datas for current version
	 * @since 0.5
	 * @return void
	 */
	function _doUpdate( $installed_version )
	{
		if ( version_compare( $installed_version, '0.5', '<' ) )
		{
			unset($this->options['engine']['tango']);
			unset($this->options['engine']['tango-image']);
			update_option('searchlink_options', $this->options );
			add_option('searchlink_engines', 
				array (
					'sitesearch' => array ( 'name' => 'sitesearch', 'enabled' => true, 'engine_url' => '/?s=','label' => 'Site','aliases' => array ( 'itt', 'here' ),'image' => false,'extra-class' => ''),
					'google' => array ( 'name' => 'google', 'enabled' => true,'engine_url' => 'http://www.google.com/search?q=','label' => 'Google','aliases' => array(),'image' => false,'extra-class' => ''),
					'google-image' => array ( 'name' => 'google-image', 'enabled' => true,'engine_url' => 'http://images.google.com/images?q=','label' => 'Google Image','aliases' => array ('google-images','google-picture','google-pictures'),'image' => true,'extra-class' => 'google'),
					'yahoo' => array ( 'name' => 'yahoo', 'enabled' => true,'engine_url' => 'http://search.yahoo.com/search?p=','label' => 'Yahoo!','aliases' => array(),'image' => false,'extra-class' => ''),
					'yahoo-image' => array ( 'name' => 'yahoo-image', 'enabled' => true,'engine_url' => 'http://images.search.yahoo.com/search/images?p=','label' => 'Yahoo! Image','aliases' => array('yahoo-images','yahoo-picture','yahoo-pictures'),'image' => true,'extra-class' => 'yahoo'),
					'msn' => array( 'name' => 'msn', 'enabled' => true,'engine_url' => 'http://search.live.com/results.aspx?q=','label' => 'MSN','aliases' => 'live','image' => false,'extra-class' => ''),
					'msn-image' => array ( 'name' => 'msn-image', 'enabled' => true,'engine_url' => 'http://search.live.com/images/results.aspx?q=','label' => 'MSN Image','aliases' => array('live-image','msn-image','msn-images','live-picture','msn-picture','live-pictures','msn-pictures'),'image' => true,'extra-class' => 'msn'),
					'altavista' => array( 'name' => 'altavista', 'enabled' => true,'engine_url' => 'http://www.altavista.com/web/results?q=','label' => 'Altavista','aliases' => array(),'image' => false,'extra-class' => ''),
					'altavista-image' => array( 'name' => 'altavista-image', 'enabled' => true,'engine_url' => 'http://www.altavista.com/image/results?q=','label' => 'Altavista Image','aliases' => array('altavista-images','altavista-picture','altavista-pictures'),'image' => true,'extra-class' => 'altavista'),
					'alltheweb' => array( 'name' => 'alltheweb', 'enabled' => true,'engine_url' => 'http://www.alltheweb.com/search?q=','label' => 'AllTheWeb','aliases' => array(),'image' => false,'extra-class' => ''),
					'alltheweb-image' => array( 'name' => 'alltheweb-image', 'enabled' => true,'engine_url' => 'http://www.alltheweb.com/search?cat=img&amp;q=','label' => 'AllTheWeb Image','aliases' => array('alltheweb-images','alltheweb-picture','alltheweb-pictures'),'image' => true,'extra-class' => 'alltheweb'),
					'miner' => array( 'name' => 'miner', 'enabled' => true,'engine_url' => 'http://miner.hu/?q=','label' => 'Miner','aliases' => array(),'image' => false,'extra-class' => 'miner')
				)
			);
			update_option('searchlink_version', '0.5');
		}
	}

}
/**
 * Get Searchlink instance
 * @since 0.3
 * @return Searchlink
 */
function getSearchlink()
{
	static $Searchlink;
	if ( is_null( $Searchlink ) )
	{
		$Searchlink = new Searchlink();
	}
	return $Searchlink;
}


// Hooks
register_activation_hook(__FILE__, array('WPGeo', 'register_activation'));

// Frontend Hooks
$Searchlink = getSearchlink();
add_filter('the_content', array( $Searchlink, 'the_content' ) );
add_filter('the_excerpt', array( $Searchlink, 'the_content' ) );
add_filter('comment_text', array( $Searchlink, 'the_content' ) );

// Admin Hooks
add_action('admin_menu', array( $Searchlink, 'admin_menu' ) );

?>
